<#
.SYNOPSIS
    Eigenstaendiges Synchronisations-Skript (Portabel, Verbessert, Nummerierte Log-Rotation)
#>

Add-Type -AssemblyName System.Windows.Forms
Add-Type -AssemblyName System.Drawing

# --- NEU: Log-Ordner und Nummerierungs-Logik ---
$logFolder = Join-Path $PSScriptRoot "SyncLogs"
if (-not (Test-Path $logFolder)) { New-Item -ItemType Directory -Path $logFolder | Out-Null }

$dateStamp = Get-Date -Format "yyyyMMdd"
$counter = 1
$logFile = Join-Path $logFolder "$($env:COMPUTERNAME)_$dateStamp-$($counter.ToString('D2')).log"

# Suche nach der hoechsten Nummer fuer heute, falls vorhanden
while (Test-Path $logFile) {
    $counter++
    $logFile = Join-Path $logFolder "$($env:COMPUTERNAME)_$dateStamp-$($counter.ToString('D2')).log"
}

Function Write-Log ($message) {
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    "$timestamp - $message" | Out-File -FilePath $logFile -Append
    Write-Host "$message"
}

# --- NEU: Log-Rotation (Max 10 Dateien) ---
$existingLogs = Get-ChildItem -Path $logFolder -Filter "*.log" | Sort-Object CreationTime -Descending
if ($existingLogs.Count -gt 10) {
    for ($i = 10; $i -lt $existingLogs.Count; $i++) {
        Remove-Item -Path $existingLogs[$i].FullName
        Write-Host "Altes Log geloescht: $($existingLogs[$i].Name)"
    }
}

# ---------------------------------------------------------

function Get-Folder($msg) {
    $objForm = New-Object System.Windows.Forms.FolderBrowserDialog
    $objForm.Description = $msg
    $result = $objForm.ShowDialog()
    if ($result -eq [System.Windows.Forms.DialogResult]::OK) {
        return $objForm.SelectedPath
    }
    return $null
}

# --- 1. Ordnerauswahl ---
$folderSource = Get-Folder("Waehlen Sie den QUELLORDNER")
if (-not $folderSource) { exit }
$folderTarget = Get-Folder("Waehlen Sie den ZIELORDNER")
if (-not $folderTarget) { exit }

Write-Log "--- Start Synchronisation ---"
Write-Log "Quelle: $folderSource"
Write-Log "Ziel: $folderTarget"

# --- 2. & 3. Vergleich, Kopieren (Optimiert) & Logik ---
$sourceFiles = Get-ChildItem -Path $folderSource -Recurse -File

foreach ($file in $sourceFiles) {
    $relativePath = $file.FullName.Substring($folderSource.Length)
    $targetPath = Join-Path $folderTarget $relativePath

    # Pruefen, ob Datei existiert und unterschiedlich ist (Groesse oder Zeit)
    if (-not (Test-Path $targetPath)) {
        # Datei ist neu
        Write-Log "Kopiere neu: $relativePath"
        $targetDir = Split-Path $targetPath
        if (-not (Test-Path $targetDir)) { New-Item -ItemType Directory -Path $targetDir -Force | Out-Null }
        
        try {
            Copy-Item -Path $file.FullName -Destination $targetPath -Force
        } catch {
            Write-Log "FEHLER beim Kopieren: $relativePath - $_"
        }
    }
    elseif (($file.Length -ne (Get-Item $targetPath).Length) -or 
            ($file.LastWriteTime.ToString("yyyyMMddHHmmss") -ne (Get-Item $targetPath).LastWriteTime.ToString("yyyyMMddHHmmss"))) {
        
        # Datei ist geaendert - Abfrage fuer Update
        $caption = "Datei geaendert"
        $message = "Die Datei '$relativePath' ist in der Quelle neuer.`n`nSoll sie im ZIEL aktualisiert werden?"
        $result = [System.Windows.Forms.MessageBox]::Show($message, $caption, [System.Windows.Forms.MessageBoxButtons]::YesNo, [System.Windows.Forms.MessageBoxIcon]::Question)
        
        if ($result -eq [System.Windows.Forms.DialogResult]::Yes) {
            try {
                Copy-Item -Path $file.FullName -Destination $targetPath -Force
                Write-Log "Aktualisiert: $relativePath"
            } catch {
                Write-Log "FEHLER beim Aktualisieren: $relativePath - $_"
            }
        }
    }
}

# --- 4. Logik: Gelöschte Dateien im Ziel zur Auswahl stellen ---
$targetFiles = Get-ChildItem -Path $folderTarget -Recurse -File
foreach ($file in $targetFiles) {
    $relativePath = $file.FullName.Substring($folderTarget.Length)
    $sourcePath = Join-Path $folderSource $relativePath

    if (-not (Test-Path $sourcePath)) {
        # Datei existiert im Ziel, aber nicht in der Quelle
        $caption = "Datei geloescht in Quelle"
        $message = "Die Datei '$relativePath' wurde im Quellordner geloescht.`n`nSoll sie auch im ZIELORDNER geloescht werden?`n(Nein = Datei im Ziel behalten)"
        
        $result = [System.Windows.Forms.MessageBox]::Show($message, $caption, [System.Windows.Forms.MessageBoxButtons]::YesNoCancel, [System.Windows.Forms.MessageBoxIcon]::Question)
        
        if ($result -eq [System.Windows.Forms.DialogResult]::Yes) {
            try {
                Remove-Item -Path $file.FullName -Force
                Write-Log "Geloescht: $relativePath"
            } catch {
                Write-Log "FEHLER beim Loeschen: $relativePath - $_"
            }
        }
    }
}

Write-Log "--- Synchronisation abgeschlossen ---"
Start-Sleep -Seconds 2